// متغيّرات عامة
let chartInstance = null;
let statsInterval = null;
let currentRange = 'week';
let currentFrom = null;
let currentTo = null;

// دالة مساعدة: تنسيق العملة
function formatCurrency(val) {
  const n = Number(val) || 0;
  return `${n.toLocaleString('ar-EG')} ﷼`;
}

// دالة مساعدة: عنوان الرسم حسب الفترة
function getChartTitle() {
  switch (currentRange) {
    case 'week':  return 'التحليل المالي — آخر 7 أيام';
    case 'month': return 'التحليل المالي — آخر شهر';
    case 'year':  return 'التحليل المالي — آخر سنة';
    case 'custom':
      if (currentFrom && currentTo) return `التحليل المالي — من ${currentFrom} إلى ${currentTo}`;
      return 'التحليل المالي — فترة مخصصة';
    default: return 'التحليل المالي';
  }
}

// دالة مساعدة: تحديث واجهة عرض الفترة المختارة (إن وُجدت العناصر)
function updateRangeUI() {
  const badge = document.getElementById('selected-range-text');
  if (badge) badge.textContent = getChartTitle().replace('التحليل المالي — ', '');

  const fromDisp = document.getElementById('from-display');
  const toDisp = document.getElementById('to-display');
  if (fromDisp) fromDisp.textContent = currentFrom || '';
  if (toDisp) toDisp.textContent = currentTo || '';
}

// دالة جلب الإحصائيات (ضع فيها كودك الخاص)
function loadStats() {
  // TODO: اكتب هنا كود جلب وعرض الإحصائيات
  console.log('loadStats placeholder');
}

// دالة تحميل البيانات للرسم البياني
async function loadChartData() {
  const projectId = document.getElementById("project-id-hidden").value;
  const activeBtn = document.querySelector('.time-btn.active');
  const range     = activeBtn?.dataset.range || 'week';
  const from      = document.getElementById('custom-from')?.value || null;
  const to        = document.getElementById('custom-to')?.value || null;

  // حفظ الحالة الحالية للفترة
  currentRange = range;
  currentFrom  = from;
  currentTo    = to;

  // تحديث واجهة الفترة (إن وُجدت عناصرها)
  updateRangeUI();

  // منع الجلب إذا كانت الفترة مخصّصة بدون تاريخ كامل
  if (range === 'custom' && (!from || !to)) {
    console.warn('فترة مخصصة تتطلب من/إلى مكتملة');
    // إظهار الحقول لو مش ظاهرة
    const customC = document.getElementById('custom-date-container');
    if (customC) customC.style.display = 'flex';
    // تحديث العنوان ليوضح أن التاريخ ناقص
    updateChart([], [], [], []);
    return;
  }

  let url = `financial_data.php?project_id=${encodeURIComponent(projectId)}&range=${encodeURIComponent(range)}`;
  if (range === 'custom') {
    url += `&from=${encodeURIComponent(from)}&to=${encodeURIComponent(to)}`;
  }

  try {
    const res = await fetch(url);
    if (!res.ok) throw new Error(`HTTP ${res.status}`);
    const ct = res.headers.get("content-type") || "";
    if (!ct.includes("application/json")) {
      const txt = await res.text();
      throw new Error("❌ السيرفر رجّع HTML بدل JSON:\n" + txt);
    }
    const result = await res.json();
    updateChart(result.labels, result.revenues, result.expenses, result.profits);
  } catch (err) {
    console.error('❌ خطأ في جلب البيانات:', err);
    updateChart([], [], [], []);
  }
}

// دالة تحديث أو إنشاء الرسم البياني
function updateChart(labels, revenues, expenses, profits) {
  const canvas = document.getElementById('financialChart');
  if (!canvas) {
    console.warn('لم يتم العثور على عنصر الرسم #financialChart');
    return;
  }
  const ctx = canvas.getContext('2d');

  if (chartInstance) {
    chartInstance.destroy();
  }

  chartInstance = new Chart(ctx, {
    type: 'line',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'الإيرادات',
          data: revenues,
          borderColor: '#c5ba9bff',
          backgroundColor: '#A19A87',
          fill: false,
          tension: 0.4,
          pointRadius: 4,
          pointHoverRadius: 6,
          pointBackgroundColor: '#A19A87',
          pointBorderColor: '#A19A87'
        },
        {
          label: 'المصروفات',
          data: expenses,
          borderColor: '#000000',
          backgroundColor:'#000000',
          fill: false,
          tension: 0.4,
          pointRadius: 4,
          pointHoverRadius: 6,
          pointBackgroundColor: '#000000',
          pointBorderColor: '#000000'
        },
        {
          label: 'الأرباح',
          data: profits,
          borderColor: '#568b53ff',
          backgroundColor:'#568b53ff',
          fill: false,  
          tension: 0.4,
          pointRadius: 4,
          pointHoverRadius: 6,
          pointBackgroundColor: '#568b53ff',
          pointBorderColor: '#568b53ff'
        }
      ]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        mode: 'nearest',
        intersect: false
      },
      plugins: {
        legend: { position: 'top' },
        title: { display: true, text: getChartTitle() },
        tooltip: {
          rtl: true,
          callbacks: {
            title: function(items) {
              // عنوان التولتيب: التصنيف (الليبل على المحور السيني)
              return items && items.length ? String(items[0].label) : '';
            },
            label: function(context) {
              const label = context.dataset.label || '';
              const val = context.parsed?.y ?? 0;
              return `${label}: ${formatCurrency(val)}`;
            },
            footer: function(items) {
              if (!items || !items.length) return '';
              // إظهار فترة "من / إلى" في التذييل إذا الفترة مخصصة
              if (currentRange === 'custom' && currentFrom && currentTo) {
                return `الفترة: من ${currentFrom} إلى ${currentTo}`;
              }
              return '';
            }
          },
          footerFont: { weight: 'normal' },
          padding: 12
        }
      },
      scales: {
        y: {
          beginAtZero: true,
          title: { display: true, text: 'القيمة ب﷼' },
          ticks: {
            callback: (v) => (Number(v)).toLocaleString('ar-EG')
          },
          grid: { color: 'rgba(0,0,0,0.06)' }
        },
        x: {
          title: { display: true, text: 'الفترة الزمنية' },
          grid: { color: 'rgba(0,0,0,0.04)' }
        }
      }
    }
  });
}

// دالة عند تغيير المشروع يدوياً من الـ <select>
function handleProjectChange(selectElement) {
  const selectedId   = selectElement.value;
  const PID          = selectedId === "All" ? 0 : parseInt(selectedId, 10);
  const selectedName = selectElement.options[selectElement.selectedIndex].text;

  document.getElementById('selected-project-text').textContent = selectedName;
  document.getElementById('selected-project-id').textContent   = PID;
  document.getElementById('project-id-hidden').value           = PID;

  console.log("✅ تم اختيار المشروع:", PID, selectedName);

  // إعادة ضبط المؤقت بشكل صحيح
  if (statsInterval) clearInterval(statsInterval);
  statsInterval = setInterval(loadStats, 5000);

  loadStats();
  loadChartData();
}

// دالة اختيار المشروع برمجيّاً
function selectProject(projectId, projectName) {
  const sel = document.getElementById('selected-project');
  sel.value = projectId === 0 ? 'All' : projectId;

  document.getElementById('selected-project-text').textContent = projectName;
  document.getElementById('selected-project-id').textContent   = projectId;
  document.getElementById('project-id-hidden').value           = projectId;

  const profitsSection = document.getElementById("profits-section");
  if (profitsSection) profitsSection.style.display = "block";

  const chartSection = document.getElementById("chart-section") || document.querySelector('.chart-container');
  if (chartSection) chartSection.style.display = "block";

  if (statsInterval) clearInterval(statsInterval);
  statsInterval = setInterval(loadStats, 5000);

  loadStats();
  loadChartData();
}

// التعامل مع أزرار اختيار الفترة الزمنية
document.querySelectorAll('.time-btn').forEach(btn => {
  btn.addEventListener('click', () => {
    document.querySelectorAll('.time-btn').forEach(b => b.classList.remove('active'));
    btn.classList.add('active');

    const range = btn.dataset.range;
    const customC = document.getElementById('custom-date-container');
    if (customC) customC.style.display = range === 'custom' ? 'flex' : 'none';

    // عند اختيار custom تأكد من وجود من/إلى قبل الجلب
    if (range === 'custom') {
      const fromEl = document.getElementById('custom-from');
      const toEl   = document.getElementById('custom-to');
      if (fromEl?.value && toEl?.value) {
        loadChartData();
      } else {
        // تحديث العنوان ليوضح أن الفترة مخصصة
        currentRange = 'custom';
        currentFrom = fromEl?.value || null;
        currentTo = toEl?.value || null;
        updateRangeUI();
        updateChart([], [], [], []);
      }
    } else {
      loadChartData();
    }
  });
});

// إعادة تحميل الرسم عند تغيير حقول التاريخ المخصص
['custom-from', 'custom-to'].forEach(id => {
  const el = document.getElementById(id);
  if (el) {
    el.addEventListener('change', () => {
      document.querySelectorAll('.time-btn').forEach(b => b.classList.remove('active'));
      const customBtn = document.getElementById('custom-btn');
      if (customBtn) customBtn.classList.add('active');

      // بعد إدخال تاريخين صالحين يتم الجلب فورًا
      const fromEl = document.getElementById('custom-from');
      const toEl   = document.getElementById('custom-to');

      currentRange = 'custom';
      currentFrom = fromEl?.value || null;
      currentTo = toEl?.value || null;
      updateRangeUI();

      if (fromEl?.value && toEl?.value) {
        loadChartData();
      } else {
        updateChart([], [], [], []);
      }
    });
  }
});

// تهيئة أولية عند تحميل الصفحة
window.addEventListener('DOMContentLoaded', () => {
  // ضبط زر "أسبوع" افتراضياً
  document.querySelectorAll('.time-btn').forEach(b => b.classList.remove('active'));
  const weekBtn = document.querySelector('.time-btn[data-range="week"]');
  if (weekBtn) weekBtn.classList.add('active');

  // إخفاء التاريخ المخصص
  const customC = document.getElementById('custom-date-container');
  if (customC) customC.style.display = 'none';

  // ضبط مشروع "الكل" افتراضياً
  const sel = document.getElementById('selected-project');
  if (sel) {
    sel.value = 'All';
    const selText = sel.options[sel.selectedIndex]?.text || 'الكل';
    document.getElementById('selected-project-text').textContent = selText;
  }
  document.getElementById('selected-project-id').textContent   = 0;
  document.getElementById('project-id-hidden').value           = 0;

  // بدء المؤقت
  if (statsInterval) clearInterval(statsInterval);
  statsInterval = setInterval(loadStats, 5000);

  loadStats();
  loadChartData();
});
